import argparse
import datetime
import smtplib
from email.mime.multipart import MIMEMultipart
from email.mime.text import MIMEText

from google.cloud import billing

#!/usr/bin/env python3
"""
GCP Cost Monitoring Tool (Basic Version)
A simple script to monitor GCP billing and send email alerts when costs exceed thresholds.
"""


# Set up command-line arguments
parser = argparse.ArgumentParser(description="Monitor GCP costs and send email alerts")
parser.add_argument("--project_id", required=True, help="GCP Project ID")
parser.add_argument("--billing_account", required=True, help="Billing Account ID")
parser.add_argument(
    "--threshold",
    type=float,
    default=100.0,
    help="Cost threshold in USD (default: 100)",
)
parser.add_argument("--email", required=True, help="Email address to send alerts to")
parser.add_argument(
    "--smtp_server", default="smtp.gmail.com", help="SMTP server for sending emails"
)
parser.add_argument("--smtp_port", type=int, default=587, help="SMTP port")
parser.add_argument("--smtp_user", required=True, help="SMTP username")
parser.add_argument("--smtp_password", required=True, help="SMTP password")
args = parser.parse_args()


def get_month_cost(client, billing_account):
    """Get the current month's cost for the billing account."""
    today = datetime.date.today()
    start_date = datetime.date(today.year, today.month, 1)

    request = {
        "aggregation": {
            "aggregation_type": "TOTAL",
            "alignment_period": {"seconds": 86400},  # Daily
        },
        "date_range": {
            "start_date": {
                "year": start_date.year,
                "month": start_date.month,
                "day": start_date.day,
            },
            "end_date": {"year": today.year, "month": today.month, "day": today.day},
        },
    }

    response = client.get_billing_info(
        name=f"billingAccounts/{billing_account}", request=request
    )

    # Extract total cost from response
    total_cost = 0.0
    for item in response.billing_info:
        if hasattr(item, "cost") and hasattr(item.cost, "amount"):
            total_cost += item.cost.amount

    return total_cost


def send_email_alert(
    to_email,
    cost,
    threshold,
    project_id,
    smtp_server,
    smtp_port,
    smtp_user,
    smtp_password,
):
    """Send an email alert when cost exceeds threshold."""

    # Prepare email content
    msg = MIMEMultipart()
    msg["From"] = smtp_user
    msg["To"] = to_email
    msg["Subject"] = f"⚠️ GCP Cost Alert: Project {project_id} exceeded ${threshold}"

    body = f"""
    <html>
    <body>
        <h2>GCP Cost Alert</h2>
        <p>Your Google Cloud Platform project <strong>{project_id}</strong> has exceeded the cost th \     \    reshold.</p>        <ul>
            <li><strong>Current Cost:</strong> ${cost:.2f}</li>
            <li><strong>Threshold:</strong> ${threshold:.2f}</li>
            <li><strong>Difference:</strong> ${cost - threshold:.2f}</li>
        </ul>

        <p>Please review your GCP resources and adjust as needed.</p>

        <p>This is an automated message from your GCP Cost Monitoring Tool.</p>
    </body>
    </html>
    """

    msg.attach(MIMEText(body, "html"))

    # Connect to SMTP server and send email
    try:
        server = smtplib.SMTP(smtp_server, smtp_port)
        server.starttls()
        server.login(smtp_user, smtp_password)
        server.send_message(msg)
        server.quit()
        print(f"Alert email sent to {to_email}")
    except Exception as e:
        print(f"Error sending email: {e}")


def main():
    # Set up GCP authentication
    # Make sure to authenticate via gcloud auth application-default login
    # or set GOOGLE_APPLICATION_CREDENTIALS environment variable

    try:
        # Create client
        client = billing.CloudBillingClient()

        # Get cost
        cost = get_month_cost(client, args.billing_account)
        print(f"Current month cost: ${cost:.2f}")

        # Check threshold
        if cost > args.threshold:
            print(f"Cost threshold exceeded: ${cost:.2f} > ${args.threshold:.2f}")
            send_email_alert(
                args.email,
                cost,
                args.threshold,
                args.project_id,
                args.smtp_server,
                args.smtp_port,
                args.smtp_user,
                args.smtp_password,
            )
        else:
            print(f"Cost within threshold: ${cost:.2f} <= ${args.threshold:.2f}")

    except Exception as e:
        print(f"Error monitoring GCP costs: {e}")


if __name__ == "__main__":
    main()
